// Photoshop Crop & Export Script (Full sizes + Duplicate-safe + Orientation-safe)

// ------------------------------------------------------
// CROP SIZE DEFINITIONS (full list)
 // ------------------------------------------------------
var cropSizes = {
    "Standard": [
        {name: "7x5", width: 7, height: 5},
        {name: "7.5x5", width: 7.5, height: 5},
        {name: "8x6", width: 8, height: 6},
        {name: "9x6", width: 9, height: 6},
        {name: "10x7", width: 10, height: 7},
        {name: "10x8", width: 10, height: 8},
        {name: "12x8", width: 12, height: 8},
        {name: "12x9", width: 12, height: 9},
        {name: "12x10", width: 12, height: 10},
        {name: "14x10", width: 14, height: 10},
        {name: "14x11", width: 14, height: 11},
        {name: "15x10", width: 15, height: 10},
        {name: "16x12", width: 16, height: 12},
        {name: "18x12", width: 18, height: 12},
        {name: "20x10", width: 20, height: 10},
        {name: "20x16", width: 20, height: 16},
        {name: "24x16", width: 24, height: 16},
        {name: "24x20", width: 24, height: 20},
        {name: "25x18", width: 25, height: 18},
        {name: "30x20", width: 30, height: 20},
        {name: "30x22", width: 30, height: 22},
        {name: "30x24", width: 30, height: 24},
        {name: "33x24", width: 33, height: 24},
        {name: "36x24", width: 36, height: 24},
        {name: "36x26", width: 36, height: 26},
        {name: "40x30", width: 40, height: 30},
        {name: "60x40", width: 60, height: 40}
    ],
    "A Sizes": [
        {name: "A5", width: 8.27, height: 5.83},
        {name: "A4", width: 11.69, height: 8.27},
        {name: "A3", width: 16.54, height: 11.69},
        {name: "A2", width: 23.39, height: 16.54},
        {name: "A1", width: 33.11, height: 23.39},
        {name: "A0", width: 46.81, height: 33.11}
    ],
    "Square": [
        {name: "6x6", width: 6, height: 6},
        {name: "8x8", width: 8, height: 8},
        {name: "10x10", width: 10, height: 10},
        {name: "11x11", width: 11, height: 11},
        {name: "12x12", width: 12, height: 12},
        {name: "16x16", width: 16, height: 16},
        {name: "20x20", width: 20, height: 20},
        {name: "24x24", width: 24, height: 24},
        {name: "30x30", width: 30, height: 30},
        {name: "40x40", width: 40, height: 40}
    ],
    "Panoramic": [
        {name: "10x5", width: 10, height: 5},
        {name: "12x5", width: 12, height: 5},
        {name: "14x8", width: 14, height: 8},
        {name: "15x5", width: 15, height: 5},
        {name: "15x6", width: 15, height: 6},
        {name: "17x6", width: 17, height: 6},
        {name: "18x8", width: 18, height: 8},
        {name: "20x8", width: 20, height: 8},
        {name: "30x10", width: 30, height: 10},
        {name: "32x16", width: 32, height: 16},
        {name: "40x20", width: 40, height: 20},
        {name: "48x16", width: 48, height: 16},
        {name: "60x20", width: 60, height: 20},
        {name: "60x30", width: 60, height: 30}
    ]
};

// Border options
var borderOptions = [
    {name: "No Border", size: 0},
    {name: "1/4 inch Border", size: 0.25},
    {name: "1/2 inch Border", size: 0.5},
    {name: "1 inch Border", size: 1},
    {name: "2 inch Border", size: 2}
];

// ------------------------------------------------------
// MAIN
// ------------------------------------------------------
function main() {
    alert("Please select the folder containing the images you want to process.");
    var sourceFolder = Folder.selectDialog("Select the folder containing images to crop:");
    if (!sourceFolder) return;

    alert("Now, please select (or create) the folder where the processed images will be saved.");
    var outputFolder = Folder.selectDialog("Select the folder to save processed images:");
    if (!outputFolder) return;

    var selection = showCropDialog();
    if (selection == null) return;

    var selectedSize = selection.size;   // object {name, width, height}
    var borderSize = selection.border;   // inches

    // Build safe file list (only Files with valid extensions)
    var rawFiles = sourceFolder.getFiles();
    var files = [];
    for (var k = 0; k < rawFiles.length; k++) {
        var f = rawFiles[k];
        if (f instanceof File && /\.(jpg|jpeg|tif|tiff|png|psd)$/i.test(f.name)) {
            files.push(f);
        }
    }

    if (files.length === 0) {
        alert("No compatible images found in the selected folder.");
        return;
    }

    // Lock output label (use selected dimensions as the label)
    var outputLabel = selectedSize.width + "x" + selectedSize.height;

    // Prevent duplicates (keyed by original filename base)
    var processedNames = {};

    // Set ruler units to inches for calculations
    var originalUnits = app.preferences.rulerUnits;
    app.preferences.rulerUnits = Units.INCHES;

    for (var i = 0; i < files.length; i++) {
        var file = files[i];
        var originalBaseName = file.name.replace(/\.[^\.]+$/, "");

        // Skip if we've already processed this base name (safety)
        if (processedNames[originalBaseName]) continue;
        processedNames[originalBaseName] = true;

        try {
            var doc = app.open(file);

            // Read original orientation BEFORE any changes
            var origW = doc.width.as("in");
            var origH = doc.height.as("in");
            var origIsLandscape = origW >= origH;

            // Convert to 300 DPI (does not change pixel aspect ratio)
            doc.resizeImage(null, null, 300, ResampleMethod.BICUBIC);

            // Determine target print orientation WITHOUT altering selected size permanently
            var targetW = selectedSize.width;
            var targetH = selectedSize.height;

            // If original orientation differs, swap target dims so final orientation matches original
            if (origIsLandscape && targetW < targetH) {
                var t = targetW; targetW = targetH; targetH = t;
            } else if (!origIsLandscape && targetH < targetW) {
                var t2 = targetW; targetW = targetH; targetH = t2;
            }

            // Calculate image area accounting for border on each side
            var imageAreaW = targetW - (borderSize * 2);
            var imageAreaH = targetH - (borderSize * 2);

            // Perform centered crop and resize to image area
            cropImage(doc, imageAreaW, imageAreaH);

            // Add border if requested (final canvas = targetW x targetH)
            if (borderSize > 0) {
                addWhiteBorder(doc, borderSize, targetW, targetH);
            }

            // Convert to sRGB profile
            convertToSRGB(doc);

            // Save - use originalBaseName for filename to avoid doc.name changes creating duplicates
            saveAsJPG(doc, originalBaseName, outputLabel, borderSize, outputFolder);

            // Close without saving changes to the PSD
            doc.close(SaveOptions.DONOTSAVECHANGES);

        } catch (e) {
            alert("Error with file: " + file.name + "\n" + e.message);
        }
    }

    // Restore ruler units
    app.preferences.rulerUnits = originalUnits;

    alert("All images processed successfully.\nSaved in: " + outputFolder.fsName);
}

// ------------------------------------------------------
// CROP SIZE DIALOG WITH BORDER OPTIONS
// ------------------------------------------------------
function showCropDialog() {
    var dialog = new Window("dialog", "Select Crop Size and Border");
    dialog.orientation = "column";
    dialog.alignChildren = "fill";
    dialog.preferredSize.width = 420;

    // Category
    var catGroup = dialog.add("group");
    catGroup.add("statictext", undefined, "Category:");
    var catDrop = catGroup.add("dropdownlist", undefined, []);

    for (var cat in cropSizes) {
        if (cropSizes.hasOwnProperty(cat)) {
            catDrop.add("item", cat);
        }
    }
    catDrop.selection = 0;

    // Size list
    var sizeGroup = dialog.add("group");
    sizeGroup.add("statictext", undefined, "Size:");
    var sizeDrop = sizeGroup.add("dropdownlist", undefined, []);
    sizeDrop.preferredSize.width = 320;

    // Border options
    var borderGroup = dialog.add("group");
    borderGroup.add("statictext", undefined, "Border:");
    var borderDrop = borderGroup.add("dropdownlist", undefined, []);
    borderDrop.preferredSize.width = 160;

    for (var i = 0; i < borderOptions.length; i++) {
        borderDrop.add("item", borderOptions[i].name);
    }
    borderDrop.selection = 0;

    function inchesToMM(x) { return Math.round(x * 25.4); }

    function updateSizes() {
        sizeDrop.removeAll();
        var list = cropSizes[catDrop.selection.text];

        for (var j = 0; j < list.length; j++) {
            var w = list[j].width;
            var h = list[j].height;

            sizeDrop.add("item",
                list[j].name + "   (" +
                inchesToMM(w) + " x " + inchesToMM(h) + " mm)"
            );
        }
        sizeDrop.selection = 0;
    }

    catDrop.onChange = updateSizes;
    updateSizes();

    // Notice text
    var notice = dialog.add(
        "edittext",
        undefined,
        "Instructions:\n" +
        "1. Select your print category and crop size.\n" +
        "2. Choose border option (optional).\n\n" +
        "Note: If your image aspect ratio differs from the chosen crop size, image cropping will occur.\n\n" +
        "Border Example: A 10x8\" print with 1\" border will have a 9x7\" image area with 1\" white border around it.",
        {multiline: true, readonly: true}
    );

    notice.preferredSize.width = 380;
    notice.preferredSize.height = 130;

    // Buttons
    var buttons = dialog.add("group");
    buttons.alignment = "center";
    buttons.add("button", undefined, "OK");
    buttons.add("button", undefined, "Cancel");

    if (dialog.show() !== 1) return null;

    return {
        size: cropSizes[catDrop.selection.text][sizeDrop.selection.index],
        border: borderOptions[borderDrop.selection.index].size
    };
}

// ------------------------------------------------------
// ORIENTATION-AWARE CENTRAL CROP (centered, ratio-respecting)
// ------------------------------------------------------
function cropImage(doc, width, height) {
    var docW = doc.width.as("in");
    var docH = doc.height.as("in");

    var targetRatio = width / height;
    var currentRatio = docW / docH;

    var left, top, right, bottom;

    if (currentRatio > targetRatio) {
        // image is wider than target ratio -> limit width
        var cropW = docH * targetRatio;
        left = (docW - cropW) / 2;
        right = left + cropW;
        top = 0;
        bottom = docH;
    } else {
        // image is taller than target ratio -> limit height
        var cropH = docW / targetRatio;
        top = (docH - cropH) / 2;
        bottom = top + cropH;
        left = 0;
        right = docW;
    }

    doc.crop([
        new UnitValue(left, "in"),
        new UnitValue(top, "in"),
        new UnitValue(right, "in"),
        new UnitValue(bottom, "in")
    ]);

    // Resize to exact target inches at 300 DPI
    doc.resizeImage(
        new UnitValue(width, "in"),
        new UnitValue(height, "in"),
        300,
        ResampleMethod.BICUBIC
    );
}

// ------------------------------------------------------
// ADD WHITE BORDER
// ------------------------------------------------------
function addWhiteBorder(doc, borderSize, finalWidth, finalHeight) {
    // Set background color to white
    var white = new SolidColor();
    white.rgb.red = 255;
    white.rgb.green = 255;
    white.rgb.blue = 255;
    app.backgroundColor = white;

    // Resize canvas to final dimensions (inches)
    doc.resizeCanvas(
        new UnitValue(finalWidth, "in"),
        new UnitValue(finalHeight, "in"),
        AnchorPosition.MIDDLECENTER
    );
}

// ------------------------------------------------------
// COLOR PROFILE CONVERSION
// ------------------------------------------------------
function convertToSRGB(doc) {
    // If the document already has sRGB profile, this call is safe
    try {
        doc.convertProfile("sRGB IEC61966-2.1", Intent.RELATIVECOLORIMETRIC, true, false);
    } catch (e) {
        // Some documents may not allow profile conversion; ignore and continue
    }
}

// ------------------------------------------------------
// SAVE AS JPG (use original filename base to avoid duplicate saves)
// ------------------------------------------------------
function saveAsJPG(doc, baseName, cropLabel, borderSize, outputFolder) {

    var borderSuffix = "";
    if (borderSize === 0.25) borderSuffix = "_quarter-border";
    else if (borderSize === 0.5) borderSuffix = "_half-border";
    else if (borderSize === 1) borderSuffix = "_1in-border";
    else if (borderSize === 2) borderSuffix = "_2in-border";

    var fileName = baseName + "_" + cropLabel + borderSuffix + ".jpg";
    var outFile = new File(outputFolder + "/" + fileName);

    var opts = new JPEGSaveOptions();
    opts.quality = 12;
    opts.embedColorProfile = true;

    // If an identical filename already exists, overwrite it (ensures only one final copy)
    doc.saveAs(outFile, opts, true, Extension.LOWERCASE);
}

// ------------------------------------------------------
main();
